// heosutil-send-command-toipaddr  Copyright 2025  Norman Carver

// Program to send a HEOS CLI command to a HEOS device (identified by its IP address),
// read the response JSON object(s), print the response (to stdout), and close the connection.
//
// HEOS_COMMAND must be a complete HEOS CLI command:
// e.g., "heos://player/get_players"
//       "heos://player/set_play_state?pid=PLAYER_PID&state=play"
//
// HEOS_DEVICE_IP_ADDRESS must be a complete dotted quad IP addres:
// e.g., 192.168.0.150
//
// The JSON responses may be pretty-printed (HEOS prettify setting) or not.
// This program checks to be sure a complete JSON object has been read
// (by checking for matching {}'s), since socket I/O does not guaratee
// the response will be gotten in a single socket read().
// Also, HEOS devices always end each JSON response object with a newline.
//
// Some HEOS commands return info that must be retrieved from the
// associated HEOS account (e.g., Favorites and Playlists).
// These HEOS commands return *two* JSON objects instead of one:
//   1) the immediate response that indicates command is good, and working on getting info
//   2) the response containing the requested info
// Must provide the optional argument NUM_JSON_OBJECTS with value 2 with such commands,
// so then and both JSON responses will be returned.
// Note that the calling program will likely be interested only in the second object,
// and "success" in the first means only that command was valid, not that info was obtained.
//
// Since sending a HEOS command to a non-HEOS device (or non-functional HEOS device)
// might end up with the program "hanging" for a significant amount of time, this program
// sets a timer, which will cause it to terminate after the specified time, regardless of
// the status of the connection.  The time is set with the preprocessor constant TIMEOUT_SECS.
// 5secs was found in testing to be sufficient, but the value could be increased if being
// used with slow network connections.



#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdbool.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>

#define BUFF_SIZE 8192
#define TIMEOUT_SECS 5


void print_usage(void)
{
  fprintf(stderr,"Usage: heosutil-send-command-toipaddr HEOS_COMMAND HEOS_DEVICE_IP_ADDRESS [NUM_JSON_OBJECTS]\n");
}


// Prototypes:
void sigalrm_handler(int signal);




int main(int argc, char *argv[])
{
  //Check if --help:
  if (argc > 1 && strcmp(argv[1],"--help") == 0) {
    print_usage();
    exit(EXIT_SUCCESS);
  }

  //Validate call:
  if (argc < 3 || argc > 4) {
    print_usage();
    exit(EXIT_FAILURE);
  }
  
  //Get HEOS device IP address and HEOS command from CLI arguments:
  char *cmd = argv[1];
  char *ipstr = argv[2];
  int numjson = 1;
  if (argc == 4)
    numjson = atoi(argv[3]);

  //Create socket:
  int sock_fd;
  struct sockaddr_in servaddr;
  if ((sock_fd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    fprintf(stderr,"ERROR: socket() call failed: %s\n",strerror(errno));
    exit(EXIT_FAILURE);
  }

  //Set up HEOS device address struct:
  servaddr.sin_family = AF_INET;
  servaddr.sin_port = htons(1255);
  inet_aton(ipstr,&servaddr.sin_addr);

  //Setup timer to deal with connect() or target not responding:
  signal(SIGALRM,sigalrm_handler);
  alarm(TIMEOUT_SECS);

  //Connect to HEOS device:
  if (connect(sock_fd, (struct sockaddr *)&servaddr, sizeof(servaddr)) == -1) {
    fprintf(stderr,"ERROR: connect() call failed: %s\n",strerror(errno));
    exit(EXIT_FAILURE);
  }

  //Send command to HEOS device (must terminate command with newline):
  if (write(sock_fd,cmd,strlen(cmd)) == -1) {
    fprintf(stderr,"ERROR: write() call to socket failed: %s\n",strerror(errno));
    exit(EXIT_FAILURE);
  }
  if (write(sock_fd,"\n",1) == -1) {
    fprintf(stderr,"ERROR: write() call to socket failed: %s\n",strerror(errno));
    exit(EXIT_FAILURE);
  }

  //Read JSON response from HEOS device and print it out to stdout:
  char buff[BUFF_SIZE];
  ssize_t nread;
  bool complete = false;
  int brackcnt = 0;  //must maintain between while iterations
  int jsoncnt = 0;
  while (!complete) {
    if ((nread = read(sock_fd,buff,BUFF_SIZE-1)) == -1) {
      fprintf(stderr,"ERROR: read() call from socket failed: %s\n",strerror(errno));
      exit(EXIT_FAILURE);
    }

    if (write(1,buff,nread) == -1) {
      fprintf(stderr,"ERROR: write() call to stdout failed: %s\n",strerror(errno));
      exit(EXIT_FAILURE);
    }

    //Check if we have read complete JSON object or not:
    char curr = '\0';
    for (int i=0; i<nread; i++) {
      curr = buff[i];
      if (curr == '{')
        brackcnt++;
      else if (curr == '}')
        brackcnt--;
    }
    if (brackcnt == 0 && curr == '\n')
      jsoncnt++;

    //Check if NUM_JSON_OBJECTS have now been read:
    if (jsoncnt == numjson)
      complete = true;
  }

  //Close connection:
  close(sock_fd);

  return EXIT_SUCCESS;
}



// Handler for SIGALRM from timer:
void sigalrm_handler(int signal)
{
  fprintf(stderr,"ERROR: connection to HEOS_DEVICE_IP_ADDRESS timed out\n");
    exit(EXIT_FAILURE);
}

// EOF
